;;; -*- Mode:Common-Lisp; Package:TV; Fonts:(CPTFONT HL10B HL12BI HL12B CPTFONTB); Base:101; * Patch-File: T -*-

;1;;                     RESTRICTED RIGHTS LEGEND          *
;1;; Use, duplication, or disclosure by the Government is subject to*
;1;; restrictions as set forth in subdivision (b)(3)(ii) of the Rights in*
;1;; Technical Data and Computer Software clause at 52.227-7013.*
;1;;                   TEXAS INSTRUMENTS INCORPORATED.*
;1;;                            P.O. BOX 149149*
;1;;                         AUSTIN, TEXAS 78714-9149*
;1;;                             MS 2151*
;1;; Copyright (C) 1986,1987, 1988, 1989 Texas Instruments Incorporated. All rights reserved.*

#|
3Overview..

This flavor defines a vertically and horizotally scrolling mouse-sensitive window.  
The logical window is unlimited in size, has possible negative coordinates,  starts off 
the same size as the physical (inside of the visible) window, and expands as required.  
"Things" (items) that you see on the window are instances of flavors such as scrollable-text-item
and scrollable-line-item.  As much as possible is compatible with flavor basic-mouse-sensitive-items.

The overview-mixin handles Mouse-m-2 (same as Sh-c-mouse-m) clicks, and displays
an overview of the entire logical-window, with the physical window shown as a rectangle,
which can be moved with the mouse.

Interesting initable instance variables.

 from x-y-scroll-bars-mixin
   :scroll-bar-thickness - # of pixels, the default is 15
   :use-both-scroll-bars-p - if NON-NIL then both scroll bars will be drawn when the user 
                                     invokes scrolling. Default is t.
   :hor-scroll-bar-always-displayed-p - if NON-NIL the hor-scroll-bar is continuously displayed.
   :ver-scroll-bar-always-displayed-p - ditto. Default for both is nil.

 from mouse-sensitivity
   :item-type-alist - as in basic-mouse-sensitive-items
   :mouse-sensitive-types - a list of types or the keyword :all. Default is :all.

 from x-y-scrolling-mixin
   :scrolling-speed - a positive integer which is how much the screen bitblt's by, when scrolling. 
                            Default is 32. Set to 1000 or more for ZMACS style "whoosh, where am I?" scrolling.

 from ver-auto-scrolling-mixin
   :increment - how much to pop up by, when the cursor reaches the bottom of the screen.  
                Can be a pixel number or one of the keywords :whole :half :quarter.  
                     The default is :half.
Methods.

  x-y-scrolling-window supports the following user-level methods.

  :scrollable-text-item item (&key x y coordinate-type mouse-sensitive-type pre-print-item-modify-function font)
  When you want to display a scrollable item whose printed representation
  is a piece of text, use this method. This makes an instance of scrollable-text-item, 
   and tells the window about the new instance. It returns the instance, to which the 
   user can later send a variety of messages.

     item - can be any lisp object. If it is not a string then you must supply
            a pre-print-item-modify-function which takes one arg, item, and returns a string.

     x - the x position of the upper left edge of the item. If x is in physical coordinates
         (ie. relative to the inside of the window) then coordinate type should be set to
         :physical. If x is in logical coordinates then coordinate-type must be set to :logical 
         (which is the default).  The default for x is the cursor position.

     y - see x above.

     coordinate-type - one of :physical :logical.  :logical is the default.

     mouse-sensitive-type - same as type in basic-mouse-senstive-items. See the methods :set-mouse-sentitive-types
                            and :set-item-type-alist. The defualt is nil, meaning not mouse sensitive.

     pre-print-item-modify-function - a function that takes one arg, the item, and returns a string for 
                                      printing.  The default is #`identity.

     font - The default is the current font of the window.

  :scrollable-line-item from-x from-y to-x to-y &key mouse-senstitive-type coordinate-type
  When you want to display a scrollable item whose printed representation is a line,
  then use this method. It makes an instance of scrollable-line-item, tells the window 
  about the new item instance, and returns the instance.

  For a explanation of the args see scrollable-text-item above.

  :set-item-type-alist (new-alist)
    new-alist - an alist of the form for basic-mouse-senstitive-items.

  :set-mouse-sensitive-types (types)
    types - can be a subset of the types in item-type-alist or the keyword :all
            meaning all the types in the alist.  The default is :all.

  The flavors scrollable-text-item and scrollable-line-item support the following user-level methods.

   :delete-self
       The instance will erase itself, and remove itself from the window's item-list.
   :move-to args
     The args depend on what type of instance.
   :center-window-on-yourself
    No args.

  For scrollable-text-item only

   :blink (how-fast-in-60ths)
   :dont-blink
      
Program internals

There are 4 different coordinate systems used here:
1) The screen coordinates ie. 750 by 1024 roughly.
   Some things in the mouse handler are in this system.

2) The outside window coordinates, which are relative to the outside left and top
   edges of the window.  cursor-x and cursor-y are in this system.  Note however
   that the args to the :set-cursorpos method are inside coordinates (#3).

3) The inside window coordinates, which are relative to the inside edges.
   The inside edges are determined by adding the margin width to the border margin
   width plus maybe the label for the bottom edge. the functions sheet-inside-left,
   sheet-inside-bottom, etc, find the inside edges.  Most higher-level text and graphic
   functions use these coordinates. (eg. :draw-line :set-cursorpos)

4) Logical coordinates, which are boundless and possibly negative.
   They look like this:          -
                                 !
                                 !
                                 !
                      -  ----------------- +
                                 !
                                 !
                                 !
                                 +
  The instance vars x-pl-offset y-pl-offset relate the inside coordinate system of
  the window (which I call the physical system) to the logical system. Specifically
  physical + ?-pl-offset = logical.*

|#
;1;;------- basic-x-y-scrolling-window -----------------------------*

;1;; no methods or instance vars for these two flavors.*

(DEFFLAVOR 4basic-x-y-scrolling-window* ()
	   (basic-x-y-scrolling-mixin
	    overview-mixin
	    mouse-sensitivity-for-instances-mixin
	    ver-auto-scrolling-mixin
	    basic-mouse-sensitive-items-compatibility-mixin
	    fancy-drag-scrolling-mixin 
	    borders-mixin
	    label-mixin
	    graphics-mixin
	    stream-mixin
	    minimum-window
	    ))
  
(DEFFLAVOR 4basic-x-y-scrolling-mixin* ()
	   (essential-x-y-scrolling-mixin
	    some-window-extensions
	    x-y-scroll-bars-mixin))


;1;;-------- essential-x-y-scrolling-mixin -----------------------------*

(DEFVAR 4default-scrolling-speed* (IF (mx-p) 64 32) 
"2Governs how fast the screen scrolls by.  
 Set to 1000 for ZMACS style --whoosh, where am I?-- scrolling*")

(DEFVAR 4*default-x-y-scroll-bar-enabledness** t
"2The default value used to determine whether x-y scroll windows will
 have the scroll bars switched on or not.*")

(DEFFLAVOR 4essential-x-y-scrolling-mixin* 
	   ((x-pl-offset 0)   ;1; pl is physical to logical*
	    ;1; it's the conversion between the two coordinate systems. Can be negative.*
	    (y-pl-offset 0)
	    (item-list nil)
	    (logical-left-edge 0)
	    (logical-right-edge 0)	   ;1; setq'd in :after :init*
	    (logical-top-edge 0)
	    (logical-bottom-edge 0)	   ;1; setq'd in :after :init*
	    (scrolling-speed default-scrolling-speed)
	    (scroll-bars-enabled-p *default-x-y-scroll-bar-enabledness*)
	    )
	   ()
  (:settable-instance-variables
    item-list
    scrolling-speed
    logical-left-edge
    logical-right-edge
    logical-top-edge
    logical-bottom-edge
    scroll-bars-enabled-p)
  (:initable-instance-variables scrolling-speed scroll-bars-enabled-p)
  :gettable-instance-variables
  (:settable-instance-variables scroll-bars-enabled-p)
  (:default-init-plist :save-bits t
    :blinker-p nil
    :deexposed-typeout-action :permit
    :more-p nil)
  (:required-flavors minimum-window)
  (:required-methods :invoke-hor-scrolling
		     :invoke-ver-scrolling
		     :invoke-hor-scrolling-mouse-handler
		     :invoke-ver-scrolling-mouse-handler)
  (:required-instance-variables hor-scrolling-in-effect
				ver-scrolling-in-effect))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :after :init*) (&rest ignore)
  (SETQ logical-bottom-edge (SEND self :inside-height))
  (SETQ logical-right-edge (SEND self :inside-width)))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :after :change-of-size-or-margins*) (&rest ignore)
  "2Recalculate logical edges.*"
  (SETQ logical-bottom-edge (SEND self :inside-height)
        logical-right-edge (SEND self :inside-width)
	logical-top-edge 0
	logical-left-edge 0
	x-pl-offset 0
	y-pl-offset 0)
  (COND ((OR (SEND self :exposed-p)
	     (AND (EQL (SEND self :deexposed-typeout-action) :permit)
		  (SEND self :active-p)))
	 (LET ((temp item-list))
	   (SEND self :clear-window)
	   (SETQ item-list temp))
	 (DOLIST (item item-list)
	   (SEND self :expand-logical-window-maybe item))
	 (DOLIST (item item-list)
	   (SEND item :draw-self)))
	(t (DOLIST (item item-list)  ;1else better not try to draw anything.*
	     (SEND self :expand-logical-window-maybe item)))))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :after :refresh*) (&rest args)
  (IF (AND args (EQUAL (FIRST args) :use-old-bits))
      nil
      (DOLIST (item item-list)
	(SEND item :refreshed)
	(SEND item :draw-self))))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :logical-height*) ()
  (- logical-bottom-edge logical-top-edge))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :logical-width*) ()
  (- logical-right-edge logical-left-edge))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :re-initialize*) ()
  (SETQ item-list nil
	x-pl-offset 0
	y-pl-offset 0
	logical-left-edge 0
	logical-top-edge 0
	logical-right-edge
	  (IF (EQUAL :recompute logical-right-edge)
	      0
	      (SEND self :inside-width))
	logical-bottom-edge
	  (IF (EQUAL :recompute logical-bottom-edge)
	      0
	      (SEND self :inside-height))))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :scrollable-text-item*)
	   (item &key
	    (mouse-sensitive-type nil)
	    ;1; the defaults below are in pixels*
	    ;1; relative to the inside coordinate system*
	    (x (- cursor-x (sheet-inside-left self)))
	    (y (- cursor-y (sheet-inside-top self)))
	    (coordinate-type :logical)
	    (font nil)
	    (pre-print-item-modify-function #'IDENTITY)
	    )
  (LET ((item (MAKE-INSTANCE 'scrollable-text-item
			     :item item
			     :mouse-sensitive-type mouse-sensitive-type
			     :font font
			     :logical-x (IF (EQL coordinate-type :physical)
					    (+ x x-pl-offset)
					    x)
			     :logical-y (IF (EQL coordinate-type :physical)
					    (+ y y-pl-offset)
					    y)
			     :window self
			     :pre-print-item-modify-function pre-print-item-modify-function)))
    (SEND item :draw-self)
    ;1; put item on list*
    (PUSH item item-list)    
    (SEND self :expand-logical-window-maybe item)
    ;1; this will do the boxing if appropriate*
    (SEND self :mouse-moves mouse-x mouse-y)
    item))  ;1; return this*

(DEFMETHOD 4(essential-x-y-scrolling-mixin :scrollable-line-item*) (from-x from-y to-x to-y)
  "2all coordinates are in logical coordinates*"
  (LET ((item (MAKE-INSTANCE 'scrollable-line-item
			     :from-x from-x
			     :from-y from-y
                             :to-x to-x
			     :to-y to-y
                             :window self)))
    (SEND item :draw-self)
    (PUSH item item-list)
    (SEND self :expand-logical-window-maybe item)
    (SEND self :mouse-moves mouse-x mouse-y)
    item))  ;1return this*

(DEFMETHOD 4(essential-x-y-scrolling-mixin :expand-logical-window-maybe*) (item)
  (IF (OR (EQUAL :recompute logical-right-edge)
	 (EQUAL :recompute logical-bottom-edge))
     (SEND self :re-initialize)
     nil)
  (IF (OR (< (SEND item :left-edge) logical-left-edge)
	  (> (SEND item :right-edge) logical-right-edge)
	  (< (SEND item :top-edge) logical-top-edge)
	  (> (SEND item :bottom-edge) logical-bottom-edge))
      (SEND self :expand-logical-window item)))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :expand-logical-window*) (item)
  (SETQ  logical-left-edge (MIN (SEND item :left-edge) logical-left-edge))
  (SETQ  logical-right-edge (MAX (SEND item :right-edge) logical-right-edge))
  (SETQ  logical-top-edge (MIN (SEND item :top-edge) logical-top-edge))
  (SETQ  logical-bottom-edge (MAX  (SEND item :bottom-edge) logical-bottom-edge)))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :after :expand-logical-window*) (IGNORE)
  (SEND self :update-scroll-bars))

(DEFUN 4scrolling-bitblt* (window dx dy speed)
  "2Special effects. Bitblts the inside of WINDOW by dx and dy in increments of SPEED. 
 If dx or dy is positive the text moves right or down the uncovered space is erased.*"
  (ASSERT (PLUSP speed) (speed) "3speed cannot be negative*")
  ;1; if how-much is so big that we're going to scroll out into space then just clear the window.*
  (COND ((OR (> (ABS dx) (SEND window :inside-width))
	     (> (ABS dy) (SEND window :inside-height)))
	 (sheet-clear window))	   ;1; we use sheet-clear insead of :clear-window cause it is closer to bitblting,*
	                           ;1; and :clear-window might have some undesirable :after methods.*
	(t (LET* ((num-of-increments (/ (SQRT (+ (* dx dx) (* dy dy))) speed))
		  (x-increment (TRUNCATE dx num-of-increments))
		  (y-increment (TRUNCATE dy num-of-increments)))
	     (DO  ((how-much-x-left-to-do dx (- how-much-x-left-to-do x-increment))
		   (how-much-y-left-to-do dy (- how-much-y-left-to-do y-increment))
		   (how-much-x-done 0 (+ how-much-x-done x-increment))
		   (how-much-y-done 0 (+ how-much-y-done y-increment)))
		  ((AND (ZEROP how-much-x-left-to-do)(ZEROP how-much-y-left-to-do)))
	       ;1; if there's not much left, do it all in one shot.*
	       (IF (OR (< (ABS how-much-x-left-to-do) (ABS (* 2 x-increment)))
		       (< (ABS how-much-y-left-to-do) (ABS (* 2 y-increment))))
		   (SETQ x-increment how-much-x-left-to-do
			 y-increment how-much-y-left-to-do))
	       (bitblt-whole-sheet window x-increment y-increment))))))

(DEFUN 4bitblt-whole-sheet* (sheet dx dy)
  "2Bitblt sheet's inside sceen array by dx and dy, and wipe out the uncovered area*"
  (SEND sheet :bitblt-within-sheet alu-seta
	(IF (PLUSP dx) -2000 2000) ;1; the 2000 will get truncated to window size. *
	(IF (PLUSP dy) -2000 2000) ;1; making the 2000 negative will change how the bitblt is done.*
	0 0 dx dy)
  ;1; wipe out the uncovered area.*
  (UNLESS (ZEROP dx)
    (SEND sheet :draw-rectangle (ABS dx) 2000
	  (IF (PLUSP dx)	
	      0    ;1;window going right, wipe our left edge.*
	      (+ (SEND sheet :inside-width) dx)) ;1;else window going left, wipe out right edge.*
	  0
	  (sheet-erase-aluf sheet)))
  (UNLESS (ZEROP dy)
    (SEND sheet :draw-rectangle 2000 (ABS dy)
	  0
	  (IF (PLUSP dy)	
	      0    ;1;window going down, wipe our top edge.*
	      (+ (SEND sheet :inside-height) dy))
	  (sheet-erase-aluf sheet))))
		   

(DEFMETHOD 4(essential-x-y-scrolling-mixin :scroll-relative*) (x y  &optional (truncate-args-p t) (inhibit-bitblt-p nil))
    (SEND self :scroll-to (+ x-pl-offset x) (+ y-pl-offset y) truncate-args-p inhibit-bitblt-p))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :scroll-to*) (logical-x logical-y &optional (truncate-args-p t)
						       (inhibit-bitblt-p nil))
  (COND (truncate-args-p      ;1; truncate args if necessary*
	 (SETQ logical-x (MAX logical-x logical-left-edge))
	 (SETQ logical-x (MIN logical-x (- logical-right-edge (SEND self :inside-width))))
	 (SETQ logical-y (MAX logical-y logical-top-edge))
	 (SETQ logical-y (MIN logical-y (- logical-bottom-edge (SEND self :inside-height)))))
	(t ;1; else expand the logical window*
	 (SETQ logical-left-edge (MIN logical-left-edge logical-x))
	 (SETQ logical-right-edge (MAX logical-right-edge (+ logical-x (SEND self :inside-width))))
	 (SETQ logical-top-edge (MIN logical-top-edge logical-y))
	 (SETQ logical-bottom-edge (MAX logical-bottom-edge (+ logical-y (SEND self :inside-height))))))
  (COND ((AND (= logical-x x-pl-offset)
	      (= logical-y y-pl-offset)))	;1; then do nothing*
	(t (UNLESS inhibit-bitblt-p
	     (scrolling-bitblt self (- x-pl-offset logical-x) (- y-pl-offset logical-y) scrolling-speed))
	   (SEND self :scroll-cursor (- x-pl-offset logical-x) (- y-pl-offset logical-y))
	   (SETQ x-pl-offset logical-x
		 y-pl-offset logical-y)
	   (DOLIST (item item-list)
	     (SEND item :draw-self)))))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :scroll-cursor*) (dx dy)
  (SEND self :set-cursorpos
	(+ dx (- cursor-x (sheet-inside-left self)))
	(+ dy (- cursor-y (sheet-inside-top self)))))

(DEFMETHOD 4(essential-x-y-scrolling-mixin :after :scroll-to*) (&rest ignore)
  (SEND self :update-scroll-bars))
(DEFMETHOD 4(essential-x-y-scrolling-mixin :handle-mouse*) ()
  (x-y-scrolling-mouse-handler self))

;1 TAC 07-25-89 - moved up here from a lower position in this file  *
(DEFVAR 4*mx-scroll-bar-tollerance** 10)

(DEFUN 4x-y-scrolling-mouse-handler* (window 				    
				    &aux
				    (window-x-offset 0) (window-y-offset 0)
				    window-x window-y)
  "2Handles the mouse if neither scroll bar is exposed.*"
  ;1; this code pilfered from mouse-default-handler.*
  (UNLESS (SYMBOLP window)
    (MULTIPLE-VALUE-SETQ (window-x-offset window-y-offset)
      (sheet-calculate-offsets self mouse-sheet)))
  
  ;1; Be careful not to do the :update method when the who line documentation window*
  ;1; isn't there (which is the case during a window system build).*
  (WHEN (AND (BOUNDP 'who-line-documentation-window) who-line-documentation-window)
    ;1; Update who-line when entering new handlers.*
    (SEND who-line-documentation-window :update))
  
  (DO ((dx) (dy) (bu) (bd) (x) (y)
       (old-owner window-owning-mouse window-owning-mouse)
       (left-offset 0)
       (right-offset 0)
       (bottom-offset 0)
       (wait-flag nil t)
       ;1; 10 is really too fast if you have a hor scroll bar also.*
       ;1; If they haven't rebound this to their favorite value, do so.*
       (scroll-bar-max-speed (IF (= scroll-bar-max-speed 10) 3
				 (= scroll-bar-max-speed 10))))
      (mouse-reconsider)
    ;1; Wait until the mouse moves*
    (MULTIPLE-VALUE-SETQ (dx dy bd bu x y) (mouse-input wait-flag))
    ;1; If asked to reconsider, do so immediately.*
    ;1; Don't bother updating blinker since it is likely to change soon, and*
    ;1; in any case we are going to be called back shortly.*
    (WHEN mouse-reconsider (RETURN nil))
    ;1; Update console-idle time when buttons pushed*
    (UNLESS (ZEROP bd) (SETQ kbd-last-activity-time (TIME)))
    ;1; Approximate speed of the mouse in inches per second*
    (SETQ mouse-speed (/ (ISQRT (+ (* mouse-x-speed mouse-x-speed)
                                   (* mouse-y-speed mouse-y-speed)))
                         100.0s0))
    ;1; If the mouse is moving incredibly fast, flash up something to*
    ;1; help the user find it.  Thus if you can't find the mouse, you must whip it.*
    (WHEN (> mouse-speed mouse-fast-motion-speed)
      (IF mouse-fast-track-bitmap-mouse-p
	  (draw-bitmap-mouse-cursor mouse-speed)
	  ;1;ELSE*
	  (draw-mouse-fast-motion-cursor)))
    
    (SETQ window-x (- x window-x-offset)	;1 X offset of mouse within window*
          window-y (- y window-y-offset))	;1 Y offset of mouse within window*
    ;1; Consider entering the scroll bar.  [Perhaps this should be changed so that*
    ;1; it is in the move-handler rather than here.  Problem with that is LEFT-OFFSET.]*
    ;1; If there is a scroll bar and we are entering it, activate it.*
    ;1; However, the mouse must move at least a certain distance past the edge*
    ;1; of the window in order to qualify for scrolling (this is set by*
    ;1; the SCROLL-BAR-RELUCTANCE variable in the window).  Before entering*
    ;1; scroll bar, send a :MOUSE-MOVES message in order to let the window know*
    ;1; what's happening.*

    ;1; LEFT-OFFSET is how far out the left side of the window the mouse has moved,*
    ;1; or 0 if the mouse is inside the window.*
    ;1; If the window is at the left edge of the screen, MOUSE-X will not itself*
    ;1; move out the left edge of the window, but DX will.  When the mouse reaches*
    ;1; the left edge of the window, accumulate leftward motion into LEFT-OFFSET.  *
    ;1; RIGHT-OFFSET does the same thing for when the scroll-bar is on the right.*
    (COND ((<= window-x 0)
	   (SETQ left-offset  (IF (PLUSP left-offset)
				  (MAX (- left-offset dx) 1)
				  (- dx))))	
	  ((AND (TYPEP window 'sheet)
		(>= window-x (sheet-width window))
		(SEND window :send-if-handles :scroll-bar-on-right))
	   (SETQ right-offset (IF (PLUSP right-offset)
				  (MAX (+ right-offset dx) 1)
				  dx)))
	  (t (SETQ left-offset 0
		   right-offset 0)))
    ;1;do the same for bottom-offset*
    (COND ((<= (sheet-height window) window-y)
	   (SETQ bottom-offset (IF (PLUSP bottom-offset) 
				   (MAX (+ bottom-offset dy) 1)
				   dy)))
	  (t (SETQ bottom-offset 0)))
    (COND ((OR old-owner window-owning-mouse (SEND self :dragging-screen-p)))	;1 These disable scroll-bar.*
	  ((AND scroll-bar-max-speed		;1Too fast, pass right through*
		(> mouse-speed scroll-bar-max-speed)))
	  ((PLUSP left-offset)
	   (COND ((AND (SEND window :scroll-bars-enabled-p)
		       (> left-offset scroll-bar-reluctance)
		       (OR (NOT (mx-p)) (< left-offset *mx-scroll-bar-tollerance*)))
		  (SEND window :mouse-moves window-x window-y)
		  (RETURN (SEND window :invoke-ver-scrolling)))
		 (t (SETQ window-x 0))))	;1Don't escape the window yet*
	  ((PLUSP bottom-offset)
	   (COND ((AND (SEND window :scroll-bars-enabled-p)
		       (> bottom-offset scroll-bar-reluctance))
		  (SEND window :mouse-moves window-x window-y)
		  (RETURN (SEND window :invoke-hor-scrolling)))
		 (t (SETQ window-y (sheet-height window))))))	;1; Don't escape the window yet*
    
    ;1; Update the position of the mouse before checking for button clicks, so*
    ;1; that button clicks get processed with knowledge of where the mouse*
    ;1; was when the button was first clicked.  The arguments to the move handler*
    ;1; may be where the mouse was when the button was clicked, whereas the*
    ;1; mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.   *
    (SETQ mouse-warp nil)
    (SEND window  :mouse-moves window-x window-y)
    ;1; Check for all the ways of losing control of the mouse.*
    (IF (COND ;1; The move handler may have decided to warp the mouse so that it will not*
	  ;1; move out of the window.  This test is a crock but should work.*
	  (mouse-warp nil)
	  ;1; Check for mouse ceasing to be grabbed.*
	  ((EQ window t)
	   (NEQ window-owning-mouse t))
	  ;1; Check for window becoming grabbed.*
	  ((EQ window-owning-mouse t)
	   (NEQ window t))
	  ;1; Check for some other window (not above this one) being greedy.*
	  (window-owning-mouse
	   (NOT (sheet-me-or-my-kid-p window window-owning-mouse)))
	  ;1; Check for moving into a window when not in any*
	  ((NULL window)
	   (window-owning-mouse x y))
	  ;1; Check for leaving the boundaries of the current window*
	  ;1; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning*
	  ((NOT (AND (sheet-exposed-p window)
		     (>= window-x 0)
		     (<  window-x (sheet-width window))
		     (>= window-y 0)
		     (<  window-y (sheet-height window))))
	   wait-flag)
	  ;1; Check for moving into an inferior of the current window*
	  ((NEQ (lowest-sheet-under-point window window-x window-y
					  :handle-mouse :exposed)
		window)
	   t))
        ;1; Return to overseer, saving any pending button click.*
        (RETURN (mouse-defer-buttons bu bd)))
    ;1; Now process button pushes if mouse is not seized.*
    (UNLESS (OR (ZEROP bd)  old-owner)
      (FUNCALL window :mouse-buttons bd window-x window-y))))

;1;;---------- fancy-drag-scrolling-mixin ---------------------*

(DEFFLAVOR 4fancy-drag-scrolling-mixin* 
	   (lx ly start-x start-y (dragging-screen-p)) 
	   () 
  :gettable-instance-variables)

(DEFMETHOD 4(fancy-drag-scrolling-mixin :mouse-click*) (b x y)
  "2Fancy drag scrolling*"
  (WHEN (AND (= b #\mouse-m)
	     (NOT (SEND self :currently-boxed-item))
	     (= (mouse-buttons t) 2)) ;1;still holding middle*
    (SETQ lx x ly y start-x x start-y y dragging-screen-p t)))

(DEFMETHOD 4(fancy-drag-scrolling-mixin :after :mouse-moves*) (x y)
  (WHEN dragging-screen-p 
    (COND ((= (mouse-buttons t) 2)
	   (bitblt-whole-sheet self (- x lx) (- y ly))
	   (SETQ lx x ly y))
	  (t (SETQ dragging-screen-p nil)
	     (IF (mx-p) ;1;; This is a strange fix to compensate for the fact*
		 ;1;; that the *1X seems to get strange values for x and y*
		 ;1;; for this method sometimes.  I don't know why, but*
		 ;1;; if you wait until both x and y are plusp, the answer*
		 ;1;; seems to be right.*
		 (LOOP for for (dx dy bd bu lx ly)
		       = (MULTIPLE-VALUE-LIST (mouse-input nil))
		       until (AND (PLUSP lx) (PLUSP ly))
		       finally (SETQ x lx y ly))
		 nil)
	     (SEND self :scroll-relative (- start-x x) (- start-y y) nil t)))))

(DEFMETHOD 4(fancy-drag-scrolling-mixin :after :handle-mouse*) (&rest ignore)
  (WHEN dragging-screen-p
    (SETQ dragging-screen-p nil)
    (SEND self :scroll-relative (- start-x lx) (- start-y ly) nil t)))

(DEFMETHOD 4(fancy-drag-scrolling-mixin :override :who-line-documentation-string*) ()
  (UNLESS (SEND self :currently-boxed-item)
    "3R2: System Menu;  MH: Drag Scrolling*"))

;1;;--------- basic scrollable item -------------------------------------*

(DEFFLAVOR 4basic-scrollable-item* 
	   ((item nil)
	    (mouse-sensitive-type nil)		;1; nil means NOT mouse-sensitive*
	    (left-edge 0)( right-edge 0)
	    (bottom-edge 0) (top-edge 0)	;1; these are the dimensions of the text*
	    msr-left msr-right msr-bottom msr-top	;1; these are the dimensions of the msr*
	    window)				;1; which is Mouse Sensitive Region.*
	   ()
  :gettable-instance-variables
  (:initable-instance-variables mouse-sensitive-type item window)
  (:settable-instance-variables item window mouse-sensitive-type)
  (:required-methods :draw-self)
  ;1;also requires an :after :init to set all the edges.*
  )

(DEFMETHOD 4(basic-scrollable-item :draw-boxing-maybe*) (logical-mouse-x logical-mouse-y)
  (COND ((SEND self :boxing-appropriate-p logical-mouse-x logical-mouse-y)
	 ;1;then get the window's item blinker, and turn it on*
	 (LET ((item-blinker (SEND window :item-blinker))
	       ;1;; Fixes for *1X put in because it doesn't seem able to clip*
	       ;1;; blinkers at the edge of windows.*
	       (left   (IF (mx-p)
			   (MAX 0 (- msr-left (SEND window :x-pl-offset)))
			   (- msr-left (SEND window :x-pl-offset))))
	       (top    (IF (mx-p)
			   (MAX 0 (- msr-top (SEND window :y-pl-offset)))
			   (- msr-top (SEND window :y-pl-offset))))
	       (right  (IF (mx-p)
			   (MIN (SEND window :inside-width)
				(- msr-right (SEND window :x-pl-offset)))
			   (- msr-right (SEND window :x-pl-offset))))
	       (bottom (IF (mx-p)
			   (MIN (SEND window :inside-height)
				(-  msr-bottom (SEND window :y-pl-offset)))
			   (-  msr-bottom (SEND window :y-pl-offset))))
	       bwidth bheight)
	   (SETQ bwidth  (- right  left)
		 bheight (- bottom top))
           ;1; Position the blinker to the item.*
	   (blinker-set-cursorpos item-blinker left top)
                                ;1  (- LEFT (SHEET-INSIDE-LEFT window))*
				;1  (- TOP  (SHEET-INSIDE-TOP window)))*
           ;1; Change the size of the blinker to enclose the item.*
	   (blinker-set-size       item-blinker bwidth bheight)
           ;1; Turn the blinker on.*
	   (blinker-set-visibility item-blinker t))
	 t))) ;1; return t to signify boxing is drawn*

(DEFMETHOD 4(basic-scrollable-item :erase-boxing*) ()
  "2Turn the blinker off*"
  (blinker-set-visibility (SEND window :item-blinker) nil))
  
(DEFMETHOD 4(basic-scrollable-item :boxing-appropriate-p*) (logical-mouse-x logical-mouse-y)
  "2In order to qualify for boxing the item must be of a current-mouse-sensitive-type,
   and the mouse has to be over the msr (mouse sensitive region) of the item.*"
  (AND (< msr-left logical-mouse-x msr-right)
       (< msr-top logical-mouse-y msr-bottom)
       (SEND window :current-mouse-sensitve-type-p mouse-sensitive-type)))

(DEFMETHOD 4(basic-scrollable-item :delete-self*) ()
  "2Erase self, then removes self from the window's item list*"
  (SEND self :erase-self)
  (SEND window :set-item-list (DELETE self (SEND window :item-list))))

(DEFMETHOD 4(basic-scrollable-item :center-window-on-yourself*) ()
  (LET ((mid-x (TRUNCATE (+ left-edge right-edge) 2))
	(mid-y (TRUNCATE (+ top-edge bottom-edge) 2)))
  (SEND window :scroll-to
	(- mid-x (TRUNCATE (SEND window :inside-width) 2))
	(- mid-y (TRUNCATE (SEND window :inside-height) 2))
	nil)))  ;1;this NIL means expand the logical window to accomodate us*

(DEFMETHOD 4(basic-scrollable-item :width*) ()
  (- right-edge left-edge))

(DEFMETHOD 4(basic-scrollable-item :height*) ()
  (- bottom-edge top-edge))

;1;------- scrollable text item ------------------------------------*

(DEFCONSTANT 4default-boxing-offset* 3 "2How far away from a word to draw the boxing.*")

(DEFFLAVOR 4scrollable-text-item* 
	   ((logical-x 0)
	    (logical-y 0)
	    font	;1; setq'd_maybe  in :after :init*
	    (pre-print-item-modify-function #'IDENTITY)	;1; must return a string. Given item.*
	    (boxing-offset default-boxing-offset)	;1; How far away from a word to draw the boxing.*
	    (blinker))
	   (basic-scrollable-item)
  (:documentation  "3An item whose printed representation is a string of text*")
  (:gettable-instance-variables logical-x logical-y font pre-print-item-modify-function boxing-offset)
  (:initable-instance-variables logical-x logical-y font pre-print-item-modify-function boxing-offset))

;1;; Stub JPR.*
(DEFMETHOD 4(scrollable-text-item :refreshed*) ()
 nil
)

(DEFMETHOD 4(scrollable-text-item :after :init*) (&rest ignore)
  (UNLESS (AND (VARIABLE-BOUNDP font) font)
    (SETQ font (SEND window :current-font)))
  ;1; Coercion put in by JPR.*
  (SETQ font
	(ETYPECASE font
	  (font font)
	  (symbol (SYMBOL-VALUE font))
	  (number (AREF (SEND window :font-map) font))))
  (ASSERT (STRINGP (FUNCALL pre-print-item-modify-function item)) (pre-print-item-modify-function)
	  "3The pre-print-item-modify-function must result in a string when applied to item.~
          Currently it resuts in ~s*" (FUNCALL pre-print-item-modify-function item))
  (SEND self :set-edges))

(DEFMETHOD 4(scrollable-text-item :set-edges*) ()
  (UNLESS font (SETQ font (SEND window :current-font)))
  (SETQ left-edge logical-x)
  (SETQ top-edge logical-y)
  (MULTIPLE-VALUE-BIND (text-length text-height)
      (SEND window :compute-motion (FUNCALL pre-print-item-modify-function item)
	    0 nil 0 0  nil 0 nil nil nil font)
    (SETQ right-edge (+ logical-x text-length))
    (SETQ bottom-edge (+ logical-y (font-char-height font) text-height)))
  (SETQ msr-left (- left-edge boxing-offset)
	msr-right (+ right-edge boxing-offset)
	msr-bottom (+ bottom-edge boxing-offset)
	msr-top (- top-edge boxing-offset)))


(DEFMETHOD 4(scrollable-text-item :move-to*) (x y &optional inhibit-redraw-p)
  (SEND self :erase-self)
  (SETQ logical-x x
	logical-y y)
  (SEND self :set-edges)
  (SEND window :expand-logical-window-maybe self)
  (UNLESS inhibit-redraw-p (SEND self :draw-self)))

(DEFMETHOD 4(scrollable-text-item :erase-self*) ()
    (LET ((physical-x (- logical-x (SEND window :x-pl-offset)))
	  (physical-y (- logical-y (SEND window :y-pl-offset))))
      ;1; since string-out-explicit goes by outside dimensions, and physical-x y are inside dimensions, do:*
      (INCF physical-x (sheet-inside-left window))
      (INCF physical-y (sheet-inside-top window))
      (SEND window :string-out-explicit-within-region
	    (FUNCALL pre-print-item-modify-function item) physical-x physical-y font (SEND window :erase-aluf))))

(DEFMETHOD 4(scrollable-text-item :draw-self*) ()
    (LET ((physical-x (- logical-x (SEND window :x-pl-offset)))
	  (physical-y (- logical-y (SEND window :y-pl-offset))))
      ;1; since string-out-explicit goes by outside dimensions, and physical-x y are inside dimensions, do:*
      (INCF physical-x (sheet-inside-left window))
      (INCF physical-y (sheet-inside-top window))
      (SEND window :string-out-explicit-within-region
	    (FUNCALL pre-print-item-modify-function item) physical-x physical-y font (SEND window :char-aluf))))

(DEFMETHOD 4(scrollable-text-item :blink*) (&optional (how-fast-in-60ths 30))
  (LET ((physical-x (- logical-x (SEND window :x-pl-offset)))
	(physical-y (- logical-y (SEND window :y-pl-offset))))
    (COND (blinker (SEND blinker :set-cursorpos physical-x physical-y)
		    (SEND blinker :set-visibility :blink)
		    (SEND blinker :set-deselected-visibility :blink)
		    (SEND blinker :set-half-period how-fast-in-60ths))
	  (t (SETQ blinker (make-blinker window
					 'rectangular-blinker
					 :width (- right-edge left-edge)
					 :height (- bottom-edge top-edge)
					 :deselected-visibility :blink
					 :x-pos physical-x
					 :y-pos physical-y
					 :half-period how-fast-in-60ths))
	     (SEND blinker :set-visibility :blink)))))

(DEFMETHOD 4(scrollable-text-item :dont-blink*) ()
  (WHEN blinker
    (SEND blinker :set-visibility nil)
    (SEND blinker :set-deselected-visibility nil)))

(DEFMETHOD 4(scrollable-text-item :after :draw-self*) (&rest ignore)
  (WHEN (AND blinker (SEND blinker :visibility))
    (LET ((physical-x (- logical-x (SEND window :x-pl-offset)))
	  (physical-y (- logical-y (SEND window :y-pl-offset))))
      (SEND blinker :set-cursorpos physical-x physical-y))))
  
;1; ------- scrollable-line-item--------------------*

(DEFFLAVOR 4scrollable-line-item* ((from-x 0) (from-y 0)
				 (to-x 0) (to-y 0))
	   (basic-scrollable-item)
  :gettable-instance-variables
  :initable-instance-variables)

;1;; Stub JPR.*
(DEFMETHOD 4(scrollable-line-item :refreshed*) ()
  nil
)

(DEFMETHOD 4(scrollable-line-item :after :init*) (&rest ignore)
  (UNLESS item 
    (SETQ item (FORMAT nil "3A line from (~a,~a) to (~a,~a)*" from-x from-y to-x to-y)))
  (SEND self :set-edges))

(DEFCONSTANT 4line-box-size* 8 "2How big a mouse sensitive box to draw on a line*")

(DEFMETHOD 4(scrollable-line-item :set-edges*) ()
  (SETQ left-edge (MIN from-x to-x)
	right-edge (MAX from-x to-x)
	top-edge (MIN from-y to-y)
	bottom-edge (MAX from-y to-y))
  (LET ((mid-x (TRUNCATE (+ left-edge right-edge) 2))
	(mid-y (TRUNCATE (+ top-edge bottom-edge) 2)))
    (SETQ msr-left (- mid-x  line-box-size)			
	  msr-right (+ mid-x line-box-size)
	  msr-top (- mid-y line-box-size)
	  msr-bottom (+ mid-y line-box-size))))

(DEFMETHOD 4(scrollable-line-item :draw-self*) ()
  ;1; (format lisp "drawing ")*
  (LET ((physical-from-x (- from-x (SEND window :x-pl-offset)))
	(physical-from-y (- from-y (SEND window :y-pl-offset)))
	(physical-to-x (- to-x (SEND window :x-pl-offset)))
	(physical-to-y (- to-y (SEND window :y-pl-offset))))
    (SEND window :draw-line physical-from-x physical-from-y physical-to-x physical-to-y (SEND window :char-aluf))))

(DEFMETHOD 4(scrollable-line-item :erase-self*) ()
  ;1; (format lisp "erasing ")*
  (LET ((physical-from-x (- from-x (SEND window :x-pl-offset)))
	(physical-from-y (- from-y (SEND window :y-pl-offset)))
	(physical-to-x (- to-x (SEND window :x-pl-offset)))
	(physical-to-y (- to-y (SEND window :y-pl-offset))))
    (SEND window :draw-line physical-from-x physical-from-y physical-to-x physical-to-y (SEND window :erase-aluf))))

(DEFMETHOD 4(scrollable-line-item :move-to*) (new-from-x new-from-y new-to-x new-to-y &optional inhibit-redraw-p)
  (SEND self :erase-self)
  (SETQ from-x new-from-x
	from-y new-from-y
	to-x new-to-x
	to-y new-to-y)
  (SEND self :set-edges)
  (SEND window :expand-logical-window-maybe self)
  (UNLESS inhibit-redraw-p (SEND self :draw-self)))

;1;------ window extensions ----------------------------------------*

(DEFFLAVOR 4some-window-extensions* () ())

(DEFMETHOD 4(some-window-extensions :mouse-warp*) (x y)
  "2X and Y are relative to the outside of self. The pane of a frame is an inferior of the pane, 
   so if you send the pane an :x-offset message you get relative to the frame.  But mouse-warp
   is relative the real tv screen.  In order to warp to a location relative to the pane, you've
   got to get the x-offset for the pane (relative to the frame) and for the frame (relative
   to the screeen). There might be N superiors.*"
    (mouse-warp (+ x (SEND self :real-x-offset)) (+ y (SEND self :real-y-offset))))

(DEFMETHOD 4(some-window-extensions :real-x-offset*) ()
  "2X and Y are relative to the outside of self.  The pane of a frame is an inferior of the pane, 
   so if you send the pane an :x-offset message you get relative to the frame.  But mouse-warp
   is relative the real tv screen.  In order to warp to a location relative to the pane, you've
   got to get the x-offset for the pane (relative to the frame) and for the frame (relative
   to the screeen). There might be N superiors.*"
  (LET ((real-x-offset 0))
    (DO ((superior self (SEND superior :superior)))
	((NOT superior) real-x-offset)
      (INCF real-x-offset (SEND superior :x-offset)))))

(DEFMETHOD 4(some-window-extensions :real-y-offset*) ()
  "2X and Y are relative to the outside of self. The pane of a frame is an inferior of the pane, 
   so if you send the pane an :x-offset message you get relative to the frame.  But mouse-warp
   is relative the real tv screen.  In order to warp to a location relative to the pane, you've
   got to get the x-offset for the pane (relative to the frame) and for the frame (relative
   to the screeen). There might be N superiors.*"
  (LET ((real-y-offset 0))
    (DO ((superior self (SEND superior :superior)))
	((NOT superior) real-y-offset)
      (INCF real-y-offset (SEND superior :y-offset)))))

(DEFMETHOD 4(some-window-extensions :string-out-explicit-within-region*)
	   (STRING start-x start-y font alu 
	    &optional
	    (min-x (sheet-inside-left self))
	    (min-y (sheet-inside-top self))
	    (max-x (sheet-inside-right self))
	    (max-y (sheet-inside-bottom self))	      
	    (start 0) end (multi-line-line-height (+ 2 (font-char-height font))))
  "2Similar to :string-out-explicit except that this will only draw the chars
   if they are in the region defined by min-x min-y max-x max-y.  Basically
   this is a generalization of :string-out-explicit's x-limit y-limit.*"
  ;1; if the text is right of, on top of, or below the region then stop here*
  (WHEN (AND (<= start-x max-x)
	     (<= start-y max-y)
	     (>= start-y min-y))
    (sheet-string-out-explicit-within-region-1 self string start-x start-y
					       min-x min-y max-x max-y
					       font alu
					       start end multi-line-line-height)))

(DEFUN 4sheet-string-out-explicit-within-region-1*
       (sheet string start-x y
	min-x min-y
	xlim ylim font alu
	&optional (start 0) (end nil)
	multi-line-line-height
	&aux fit fwt lkt
	(x start-x))
  "2Output STRING on SHEET without using SHEET's cursor, font, etc.
   Output starts at cursor position START-X, Y if both are greater than
   min-x min-y. SHEET's cursor is not moved. 
   Output stops if x-position XLIM or y-position YLIM is reached.

   Font FONT is used, and alu-function ALU.
   START and END specify a portion of STRING to be used.
   MULTI-LINE-LINE-HEIGHT is how far to move down for Return
   characters; Return also moves back to x-position START-X.
   NIL means output <Return> with a lozenge.

  All position arguments are relative to SHEET's outside edges.*"
  
  (DECLARE (VALUES final-x final-y final-index))
  (coerce-font font sheet)
  (SETQ fit (font-indexing-table   font)
	fwt (font-char-width-table font)
	lkt (font-left-kern-table  font))
  (OR xlim (SETQ xlim (sheet-width sheet)))
  (prepare-sheet (sheet)
    (DO ((i start (1+ i))
	 (n (OR end (ARRAY-ACTIVE-LENGTH string)))
	 (width (font-char-width font))
	 (ch))
	((>= i n) (VALUES x y i))
      (SETQ ch (AREF string i))
      (COND ((AND multi-line-line-height (MEMBER ch '(#\newline #\Return) :test #'CHAR-EQUAL))
	     (SETQ x start-x
                   y (+ y multi-line-line-height))
	     (IF (AND ylim (> (+ y multi-line-line-height) ylim))
		 (RETURN x y i)))
	    ((>= ch 200)
	     (LET* ((STRING (STRING
                              (OR
                                (CAR (RASSOC ch si::xr-special-character-names))
                                ;1; If there is no name for the character*
                                ;1; then print out its octal character number.*
                                (FORMAT nil "3~3O*" ch))))
		    (nx (+ x (lozenged-string-geometry string))))
	       (IF (> nx xlim) (RETURN x y i))
	       (sheet-display-lozenged-string-internal sheet string
						       x (1+ y) xlim alu)
	       (SETQ x nx)))
	    (t (IF fwt (SETQ width (OR (AREF fwt ch) (font-char-width font))))
	       (IF (> (+ x width) xlim) (RETURN x y i))
	       (WHEN (AND (<= min-x x)		;1; This extra "when", and the lambda list, are the only things*
			  (<= min-y y)		;1; different from SHEET-STRING-OUT-EXPLICIT-1*
			  (>= ylim (+ y (font-char-height font))))
		 (draw-char font ch
			    (IF lkt
				(- x (AREF lkt ch))
				;1;ELSE*
				x)
			    y alu sheet))
	       (SETQ x (+ x width)))))))



;1;;*
;1;;   Reed Hastings     Hastings@sumex.stanford.edu     January 1987*
;1;;*
;1;;   The flavor defs and methods for flavor x-y-scroll-bars-mixin.*
;1;;   This flavor sets up horizontal and vertical scroll bars.*
;1;;  *
;1;;   Mix this in with a window if that window:*
;1;;   *
;1;;   a) has a mouse handler that sends self :invoke-ver-scrolling and :invoke-hor-scrolling when appropriate;*
;1;;   b) has the required methods and instance variables below.  They are mostly*
;1;;      used to calculate the correct scroll bar length and position;*
;1;;   c) handles the :scroll-to message which has two args: new-x-position and new-y-position, which are*
;1;;      in pixels on the logical screen, and might be negative;  the handler should truncate args into range.*
;1;;   d) handles the :scroll-relative message which has two args: how-many-pixels and direction.*
;1;;   *
;1;;   *
;1;;   Overwiew.*
;1;;   *
;1;;   a typical (vertical) scrolling action would go like this:  the window's mouse handler would notice that the mouse*
;1;;   is requesting scrolling (by slowly leaving the top probably). It sends an :invoke-ver-scrolling*
;1;;   message.  The method below would draw the scroll bar, alter the mouse cursor character, and*
;1;;   call the ver-scrolling-mouse-handler.  This handler will keep the mouse in the correct region, sending*
;1;;   :mouse-moves-scroll  and :mouse-buttons-scroll messages as appropriate. After sending each :mouse-buttons-scroll*
;1;;   message it will redraw the scroll bar to reflect the (possibly) new situation.  This mouse handler will*
;1;;   return when the mouse legally moves out of the scroll bar region, or when the mouse is seized (like by*
;1;;   some other window covering our scrolling window.)  After the mouse handler returns, the *
;1;;   :invoke-ver-scrolling method will erase the scroll bar, and reset the mouse cursor character, before*
;1;;   returning (presumably to the window's regular mouse handler).*
;1;;*
;1;;*
;1;;   Notes.*
;1;;*
;1;;   The code was pinched from the TI system code. Mostly from the mouse-default-handler.*
;1;;*
;1;;*

(DEFCONSTANT  4default-scroll-bar-thickness* 15)
							   
(DEFFLAVOR 4x-y-scroll-bars-mixin* 
	   ((hor-scrolling-in-effect nil)      ;1; a flag for general use*
	    (ver-scrolling-in-effect nil)      ;1; a flag for general use*
	    (ver-scroll-bar nil)	       ;1; a record of the scroll bar dimensions, nil if not drawn.*
	    (hor-scroll-bar nil)	       ;1; a record of the scroll bar dimensions, nil if not drawn.*
	    (scroll-bar-thickness default-scroll-bar-thickness)
	    (use-both-scroll-bars-p t)         ;1; if t, both scroll bars are drawn when scrolling invoked.*	
	    (hor-scroll-bar-always-displayed-p nil)
	    (ver-scroll-bar-always-displayed-p nil))
	   ()
  (:required-instance-variables logical-left-edge x-pl-offset logical-top-edge y-pl-offset logical-bottom-edge)
  (:required-methods :scroll-relative :scroll-to :logical-width :logical-height :inside-width :inside-height)
  (:required-flavors minimum-window)
  (:initable-instance-variables scroll-bar-thickness
				use-both-scroll-bars-p
				ver-scroll-bar-always-displayed-p
				hor-scroll-bar-always-displayed-p)
  (:settable-instance-variables scroll-bar-thickness
				use-both-scroll-bars-p
				ver-scroll-bar-always-displayed-p
				hor-scroll-bar-always-displayed-p)) 

(DEFMETHOD 4(x-y-scroll-bars-mixin :after :init*) (&rest ignore)
  "2give room for scroll bars on left and top edges.*"
  (SEND self :set-border-margin-width (+ scroll-bar-thickness 1)))

(DEFMETHOD 4(x-y-scroll-bars-mixin :after :expose*) (&rest ignore)
  (WHEN hor-scroll-bar-always-displayed-p (SEND self :draw-hor-scroll-bar))
  (WHEN ver-scroll-bar-always-displayed-p (SEND self :draw-ver-scroll-bar)))

(DEFMETHOD 4(x-y-scroll-bars-mixin :before :change-of-size-or-margins*) (&rest ignore)
  "2We have to get rid of them because they will change in length upredictably,
   and we won't know how to erase them.*"
  (SEND self :erase-hor-scroll-bar)
  (SEND self :erase-ver-scroll-bar))

(DEFMETHOD 4(x-y-scroll-bars-mixin :override :who-line-documentation-string*) ()
  (COND (ver-scrolling-in-effect self
	 (IF (= (SEND self :logical-height) (SEND self :inside-height))
	     "3Nothing to scroll to vertically.  There is nothing above or below.*"
	     '(:mouse-l-1 "3this place to top*"
			  :mouse-l-2 "3this place to bottom*"
			  :mouse-m-1 "3percentage-wise*"
			  :mouse-r-1 "3top to this place*"
			  :mouse-r-2 "3bottom to this place*")))
	(hor-scrolling-in-effect
	 (IF (= (SEND self :logical-width) (SEND self :inside-width))
	     "3Nothing to scroll to horizontally.  There is nothing left or right.*"
	     '(:mouse-l-1 "3this place to left edge*"
			  :mouse-l-2 "3this place to right edge*"
			  :mouse-m-1 "3percentage-wise*"
			  :mouse-r-1 "3left edge to this place*"
			  :mouse-r-2 "3right edge to this place.*")))))

;1; the mouse char has already been changed so just pass on the :mouse-moves message.*
(DEFMETHOD 4(x-y-scroll-bars-mixin :mouse-moves-scroll*) (x y)
  (SEND self :mouse-moves x y))

(DEFMETHOD 4(x-y-scroll-bars-mixin :mouse-buttons-scroll*) (bd x y)
  "2decides how much to scroll in what direction*"
  ;1; bd is a mask of what button is down. mouse-button-encode looks for double clicks or whatever.*
  (LET ((button (mouse-character-button-encode bd)))
    (COND (hor-scrolling-in-effect
	   (CASE button
		 (#\mouse-l (SEND self :scroll-relative x 0))
		 (#\mouse-l-2 (SEND self :scroll-relative (- x width) 0))
		 (#\mouse-r (SEND self :scroll-relative (- x) 0))
		 (#\mouse-r-2 (SEND self :scroll-relative (- width x) 0))
		 (#\mouse-m (SEND self :scroll-to
				  (TRUNCATE (+ logical-left-edge (* (SEND self :logical-width)
					       (/ x width))))
				  y-pl-offset))
		 (otherwise (BEEP))))
	  (ver-scrolling-in-effect
	   (CASE button
		 (#\mouse-l (SEND self :scroll-relative 0 y))
		 (#\mouse-l-2 (SEND self :scroll-relative 0 (- y height)))
		 (#\mouse-r (SEND self :scroll-relative 0 (- y)))
		 (#\mouse-r-2 (SEND self :scroll-relative 0 (- height y)))
		 (#\mouse-m (SEND self :scroll-to
				  x-pl-offset
				  (TRUNCATE (+ logical-top-edge (* (SEND self :logical-height)
					       (/ y height))))))
		 (otherwise (BEEP))))
	  (t (CERROR "3ignore safely*" "3neither scroll bar is in*")))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :update-scroll-bars*) ()
  (WHEN ver-scroll-bar
	 (SEND self :erase-ver-scroll-bar)
	 (SEND self :draw-ver-scroll-bar))
  (WHEN hor-scroll-bar
	 (SEND self :erase-hor-scroll-bar)
	 (SEND self :draw-hor-scroll-bar)))

;1;-------- vertical scrolling code. ------ horizontal is symmetric.*

(DEFMETHOD 4(x-y-scroll-bars-mixin :invoke-ver-scrolling*) ()
  "2Called by the window's mouse handler when the mouse enters the vertical scroll bar.
   Returns when the mouse leaves the scrolling area.*"
  ;1; Give feedback by changing mouse cursor before calling SCROLL-BAR-DRAW, which pages a lot*
  (SEND self ':set-mouse-position (TRUNCATE scroll-bar-width 2) nil)
  ;1; Change the mouse to a double-headed up-and-down arrow.*
  (mouse-set-blinker-definition ':character 0 7 ':on
				':set-character 4)
  ;1; Draw the scroll bar(s)*
  ;1; We don't care if there already drawn, cause :draw-ver-scroll-bar will do the right thing.*
  (SEND self :draw-ver-scroll-bar)
  (WHEN use-both-scroll-bars-p (SEND self :draw-hor-scroll-bar))
  
  ;1; let this handler run. It will return when the mouse leaves the scrolling area.*
  ;1; while it is running set the flag*
  (SETQ ver-scrolling-in-effect t)
  (SEND self :invoke-ver-scrolling-mouse-handler)
  (SETQ ver-scrolling-in-effect nil)
  ;1; put back blinker*
  (mouse-standard-blinker self)
  ;1; erase scroll bar*
  (UNLESS ver-scroll-bar-always-displayed-p (SEND self :erase-ver-scroll-bar))
  (UNLESS hor-scroll-bar-always-displayed-p (WHEN hor-scroll-bar (SEND self :erase-hor-scroll-bar))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :draw-ver-scroll-bar*) ()
  (WITHOUT-INTERRUPTS
    (IF (sheet-can-get-lock self)
	(ver-scroll-bar-draw)
	(PROCESS-RUN-FUNCTION "3Draw Scroll Bar*"
			      (LET-CLOSED ((self self)
					   (si:self-mapping-table si:self-mapping-table))
				  (sheet-force-access (self)
				    ;1; It is possible that the mouse moved out while we were*
				    ;1; waiting.  If this is the case, punt drawing.*
				    (AND (SEND self :ver-scrolling-in-effect)
					 (ver-scroll-bar-draw))))))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :erase-ver-scroll-bar*) ()
  (WITHOUT-INTERRUPTS
    ;1; There is this funny case where the sheet could be locked by the person waiting for us to back out.*
    ;1; For us to block here would be a disaster, so undraw the scroll bar in another process.*
    (IF (sheet-can-get-lock self)
	(sheet-force-access (self) (ver-scroll-bar-erase))
	(PROCESS-RUN-FUNCTION "3Undraw Scroll Bar*"
			      (LET-CLOSED ((self self)
					   (si:self-mapping-table si:self-mapping-table))
				  (sheet-force-access (self)
				    ;1; It is possible that the user reentered the scroll bar before this code ran.  *
				    ;1; In that case, don't actually erase it.*
				    (OR (SEND self :ver-scrolling-in-effect)
					(ver-scroll-bar-erase))))))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :invoke-ver-scrolling-mouse-handler*)
				    (&aux
				    (window-x-offset 0) (window-y-offset 0)
				    window-x window-y
				    (window self))
				    
  "2Handles the mouse if vertical scroll bar is engaged.*"
  (MULTIPLE-VALUE-SETQ (window-x-offset window-y-offset)
    (sheet-calculate-offsets self mouse-sheet))
  (DO ((dx) (dy) (bu) (bd)  (x) (y)
       (old-owner window-owning-mouse window-owning-mouse)
       (wait-flag nil t))
      (mouse-reconsider)
    ;1; Wait until the mouse moves*
    (IF (mx-p)
	;1;; Do this twice because otherwise we get bogus values.*
	(MULTIPLE-VALUE-SETQ (dx dy bd bu x y) (mouse-input nil))
	nil)
    (MULTIPLE-VALUE-SETQ (dx dy bd bu x y) (mouse-input wait-flag))
    ;1; If asked to reconsider, do so immediately.*
    ;1; Don't bother updating blinker since it is likely to change soon, and*
    ;1; in any case we are going to be called back shortly.*
    (IF mouse-reconsider (RETURN nil))

    (SETQ window-x (- x window-x-offset)	;1; X offset of mouse within window*
	  window-y (- y window-y-offset))	;1; Y offset of mouse within window*
    ;1; Approximate speed of the mouse in inches per second*
    (SETQ mouse-speed (TRUNCATE (ISQRT (+ (* mouse-x-speed mouse-x-speed)
					  (* mouse-y-speed mouse-y-speed)))
				100))
    ;1; maybe leave the scroll bar*
    (COND ((AND scroll-bar-max-exit-speed
		(> mouse-speed scroll-bar-max-exit-speed))
	   ;1; Moving like a bat, let the guy out of the scroll bar*
	   (RETURN nil))
	  ((> window-x scroll-bar-width)	        ;1; Escape out right*
	   (RETURN nil))
	  ((MINUSP window-x)			        ;1; Trying to go out left, shove back in*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-x 0)
	     (SETQ mouse-last-x (SETQ mouse-x window-x-offset))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil)))
	  ((MINUSP window-y)			        ;1; Trying to go out up, shove back in*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-y 0)
	     (SETQ mouse-last-y (SETQ mouse-y window-y-offset))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil)))
	  ((> window-y (SEND window :inside-height))	;1; Trying to go out down, shove back in.*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-y (SEND window :inside-height))
	     (SETQ mouse-last-y (SETQ mouse-y (+ window-y-offset (SEND window :inside-height))))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil))))
    ;1; Update the position of the mouse before checking for button clicks, so that button clicks*
    ;1; get processed with knowledge of where the mouse was when the button was first clicked. *
    ;1; The arguments to the move handler may be where the mouse was when the button was clicked, *
    ;1; whereas the mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.    *
    (SETQ mouse-warp nil)
    (SEND window :mouse-moves-scroll window-x window-y)
    ;1; Check for all the ways of losing control of the mouse.*
    (IF (COND ;1; The move handler may have decided to warp the mouse so that it will not*
	      ;1; move out of the window.  This test is a crock but should work.*
	  (mouse-warp nil)
	  ;1; Check for mouse ceasing to be grabbed.*
	  ((EQ window t)
	   (NEQ window-owning-mouse t))
	  ;1; Check for window becoming grabbed.*
	  ((EQ window-owning-mouse t)
	   (NEQ window t))
	  ;1; Check for some other window (not above this one) being greedy.*
	  (window-owning-mouse
	   (NOT (sheet-me-or-my-kid-p window window-owning-mouse)))
	  ;1; Check for moving into a window when not in any*
	  ((NULL window)
	   (window-owning-mouse x y))
	  ;1; Check for leaving the boundaries of the current window*
	  ;1; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning*
	  ((AND (mx-p)
		;1; We can't warp the mouse on the MX when we end up in*
		;1; a scroll bar, so be more lenient.*
		(NOT (AND (sheet-exposed-p window)
			  ;1; When the mouse is outside the window the values of window-x*
			  ;1; and window-y stay constant at the edges of the window, so we  *
			  ;1; need to calculate a certain amount of hysteresis on our own.*
			  (AND (<= y (+ window-y window-y-offset))
			       (AND (>= x (+ (- *mx-scroll-bar-tollerance*)
					     window-x window-x-offset))
				    (<= x (+ *mx-scroll-bar-tollerance*
					     window-x window-x-offset))))
			  (>= window-x (- *mx-scroll-bar-tollerance*))
			  (<  window-x (+ *mx-scroll-bar-tollerance*
					  (sheet-width window)))
			  (>= window-y (- *mx-scroll-bar-tollerance*))
			  (<  window-y (+ *mx-scroll-bar-tollerance*
					  (sheet-height window)))
			  )))
	   wait-flag)
	  ((AND (NOT (mx-p))
		(NOT (AND (sheet-exposed-p window)
			  (>= window-x 0)
			  (<  window-x (sheet-width window))
			  (>= window-y 0)
			  (<  window-y (sheet-height window)))))
	   wait-flag)
	  ;1; Check for moving into an inferior of the current window*
	  ((NEQ (lowest-sheet-under-point window window-x window-y
					  ':handle-mouse ':exposed)
		window)
	   t))
	;1; Return to overseer, saving any pending button click.*
	(RETURN (mouse-defer-buttons bu bd)))
    ;1; Now process button pushes if mouse is not seized.*
    (UNLESS (OR (ZEROP bd) old-owner)
      (FUNCALL window :mouse-buttons-scroll bd window-x window-y)
      (SEND window :update-scroll-bars))))

;1;;-------- horizontal scrolling ----- symmetric code to above vertical scrolling*

(DEFMETHOD 4(x-y-scroll-bars-mixin :invoke-hor-scrolling*) ()
  "2Called when the mouse enters the horizontal scroll bar
     Returns when the mouse leaves the scrolling area.*"
  ;1; Give feedback by changing mouse cursor before calling SCROLL-BAR-DRAW, which pages a lot*
  (SEND self ':set-mouse-position nil (- (sheet-height self) (TRUNCATE scroll-bar-width 2)))

  ;1; Change the mouse to a fat double-headed up-and-down arrow.*
  (mouse-set-blinker-definition ':character 0 7 ':on
				':set-character 5)

  ;1; Draw the scroll bar*
  (SEND self :draw-hor-scroll-bar)
  (WHEN use-both-scroll-bars-p (SEND self :draw-ver-scroll-bar))
  ;1; Let this handler run. It will return when the mouse leaves the scrolling area.*
  ;1; while it runs set this flag for the who line and maybe others*
  (SETQ hor-scrolling-in-effect t)
  (SEND self :invoke-hor-scrolling-mouse-handler)
  (SETQ hor-scrolling-in-effect nil)
  ;1; put back blinker*
  (mouse-standard-blinker self)
  ;1; erase scroll bar*
  (UNLESS hor-scroll-bar-always-displayed-p (SEND self :erase-hor-scroll-bar))
  (UNLESS ver-scroll-bar-always-displayed-p (WHEN ver-scroll-bar (SEND self :erase-ver-scroll-bar))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :draw-hor-scroll-bar*) ()
  (WITHOUT-INTERRUPTS
    (IF (sheet-can-get-lock self)
	(hor-scroll-bar-draw)
	(PROCESS-RUN-FUNCTION "3Draw Scroll Bar*"
			      (LET-CLOSED ((self self)
					   (si:self-mapping-table si:self-mapping-table))
				  (sheet-force-access (self)
				    ;1; It is possible that the mouse moved out while we were*
				    ;1; waiting.  If this is the case, punt drawing.*
				    (AND (SEND self :hor-scrolling-in-effect)
					 (hor-scroll-bar-draw))))))))

(DEFMETHOD 4(x-y-scroll-bars-mixin :erase-hor-scroll-bar*) ()
    (WITHOUT-INTERRUPTS
      ;1; There is this funny case where the sheet could be locked by the person waiting for us to back out.*
      ;1; For us to block here would be a disaster, so undraw the scroll bar in another process.*
      (IF (sheet-can-get-lock self)
	  (sheet-force-access (self) (hor-scroll-bar-erase))
	  (PROCESS-RUN-FUNCTION "3Undraw Scroll Bar*"
				(LET-CLOSED ((self self)
					   (si:self-mapping-table si:self-mapping-table))
				    (sheet-force-access (self)
				      ;1; It is possible that the user reentered the scroll bar before this code ran. *
				      ;1; In that case, don't actually erase it.*
				      (OR (SEND self :hor-scrolling-in-effect)
					  (hor-scroll-bar-erase))))))))

;1 TAC 07-25-89 - moved up - this has to occur sooner in this file for proper compilation*
;1(defvar *mx-scroll-bar-tollerance* 10)*

(DEFMETHOD 4(x-y-scroll-bars-mixin :invoke-hor-scrolling-mouse-handler*) 
				    (&aux
				    (window-x-offset 0) (window-y-offset 0)
				    window-x window-y
				    (window self))
				    
  "2Handles the mouse if horizontal scroll bar is exposed.*"
  (MULTIPLE-VALUE-SETQ (window-x-offset window-y-offset)
    (sheet-calculate-offsets self mouse-sheet)) 
  (DO ((dx) (dy) (bu) (bd)  (x) (y)
       (old-owner window-owning-mouse window-owning-mouse)
       (wait-flag nil t))
      (mouse-reconsider)
    ;1; Wait until the mouse moves*
    (MULTIPLE-VALUE-SETQ (dx dy bd bu x y) (mouse-input wait-flag))
    ;1; If asked to reconsider, do so immediately.*
    ;1; Don't bother updating blinker since it is likely to change soon, and*
    ;1; in any case we are going to be called back shortly.*
    (IF mouse-reconsider (RETURN nil))

    (SETQ window-x (- x window-x-offset)	;1; X offset of mouse within window*
	  window-y (- y window-y-offset))	;1; Y offset of mouse within window*
    ;1; Approximate speed of the mouse in inches per second*
    (SETQ mouse-speed (TRUNCATE (ISQRT (+ (* mouse-x-speed mouse-x-speed)
					  (* mouse-y-speed mouse-y-speed)))
				100))
    ;1; Maybe leave the scroll bar and maybe move the cursor in.*
    (COND ((AND scroll-bar-max-exit-speed
		(> mouse-speed scroll-bar-max-exit-speed))
	   ;1; Moving like a bat, let the guy out of the scroll bar*
	   (RETURN nil))
	  ((< window-y (- (sheet-height window) scroll-bar-width))	;1; Escape out up*
	   (RETURN nil))
	  ((> window-y 	(sheet-height window))		;1; Trying to go out down, shove back in*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-y (sheet-height window))
	     (SETQ mouse-last-y (SETQ mouse-y (+ (sheet-height window) window-y-offset)))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil)))
	  ((MINUSP window-x)			        ;1; Trying to go out left, shove back in*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-x 0)
	     (SETQ mouse-last-x (SETQ mouse-x window-x-offset))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil)))
	  ((> window-x (SEND window :inside-width))	;1; Trying to go out right, shove back in.*
	   (WITHOUT-INTERRUPTS
	     (%open-mouse-cursor)
	     (SETQ window-x (SEND window :inside-width))
	     (SETQ mouse-last-x (SETQ mouse-x (+ window-x-offset (SEND window :inside-width))))
	     (SETQ mouse-cursor-state mouse-cursor-closed-state
		   prepared-sheet nil)))
	  )
    ;1; Update the position of the mouse before checking for button clicks, so that button clicks *
    ;1; get processed with knowledge of where the mouse was when the button was first clicked. *
    ;1; The arguments to the move handler may be where the mouse was when the button was clicked,*
    ;1; whereas the mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.    *
    (SETQ mouse-warp nil)
    (SEND window :mouse-moves-scroll window-x window-y)
    ;1; Check for all the ways of losing control of the mouse.*
    (IF (COND ;1; The move handler may have decided to warp the mouse so that it will not*
	      ;1; move out of the window.  This test is a crock but should work.*
	  (mouse-warp nil)
	  ;1; Check for mouse ceasing to be grabbed.*
	  ((EQ window t)
	   (NEQ window-owning-mouse t))
	  ;1; Check for window becoming grabbed.*
	  ((EQ window-owning-mouse t)
	   (NEQ window t))
	  ;1; Check for some other window (not above this one) being greedy.*
	  (window-owning-mouse
	   (NOT (sheet-me-or-my-kid-p window window-owning-mouse)))
	  ;1; Check for moving into a window when not in any*
	  ((NULL window)
	   (window-owning-mouse x y))
	  ;1; Check for leaving the boundaries of the current window*
	  ;1; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning*
	  ((AND (mx-p)
		;1; We can't warp the mouse on the MX when we end up in*
		;1; a scroll bar, so be more lenient.*
		(NOT (AND (sheet-exposed-p window)
			  ;1; When the mouse is outside the window the values of window-x*
			  ;1; and window-y stay constant at the edges of the window, so we *
			  ;1; need to calculate a certain amount of hysteresis on our own.*
			  (OR (EQ window (window-under-mouse))
			      (AND (<= y (+ *mx-scroll-bar-tollerance*
					    window-y window-y-offset))
				   (<= x (+ *mx-scroll-bar-tollerance*
					    window-x window-x-offset)))
			  )
			  (>= window-x (- *mx-scroll-bar-tollerance*))
			  (<  window-x (+ *mx-scroll-bar-tollerance*
					  (sheet-width window)))
			  (>= window-y (- *mx-scroll-bar-tollerance*))
			  (<  window-y (+ *mx-scroll-bar-tollerance*
					  (sheet-height window))))))
	   wait-flag)
	  ((AND (NOT (mx-p))
		(NOT (AND (sheet-exposed-p window)
			  (>= window-x 0)
			  (<  window-x (sheet-width window))
			  (>= window-y 0)
			  (<  window-y (sheet-height window)))))
	   wait-flag)
	  ;1; Check for moving into an inferior of the current window*
	  ((NEQ (lowest-sheet-under-point window window-x window-y
					  ':handle-mouse ':exposed)
		window)
	   t))
	;1; Return to overseer, saving any pending button click.*
	(RETURN (mouse-defer-buttons bu bd)))
    ;1; Now process button pushes if mouse is not seized.*
    (UNLESS (OR (ZEROP bd) old-owner)
      (FUNCALL window :mouse-buttons-scroll bd window-x window-y)
      (SEND window :update-scroll-bars))))

;1; --------- scroll bar draw and erase functions ------------*
(DEFUN-METHOD 4ver-scroll-bar-draw* x-y-scroll-bars-mixin ()
  (WHEN ver-scroll-bar (SEND self :erase-ver-scroll-bar))
  (LET ((LENGTH (TRUNCATE (* height (/ (SEND self :inside-height) (SEND self :logical-height)))))
	(top (TRUNCATE (* height (/ (- y-pl-offset logical-top-edge) (SEND self :logical-height)))))) 
    (prepare-sheet (self)
      ;1; Erase anything there first.*
      (%draw-rectangle scroll-bar-thickness length 0 top  alu-andca self)
      ;1; Now we can draw the scroll bar.*
      (%draw-rectangle scroll-bar-thickness length 0 top  alu-ior self))
    ;1; Make a record of the scroll bar so it can be erased correctly*
    (SETQ ver-scroll-bar (LIST top length scroll-bar-thickness))))

(DEFUN-METHOD 4ver-scroll-bar-erase* x-y-scroll-bars-mixin ()
  (WHEN ver-scroll-bar
    (LET ((top (FIRST ver-scroll-bar))
	  (LENGTH (SECOND ver-scroll-bar))
	  (thickness (THIRD ver-scroll-bar)))
      (prepare-sheet (self)
	(%draw-rectangle thickness length 0 top  alu-andca self)
	(SEND self :refresh-margins)
	(SETQ ver-scroll-bar nil)))))

(DEFUN-METHOD 4hor-scroll-bar-draw* x-y-scroll-bars-mixin ()
  (WHEN hor-scroll-bar (SEND self :erase-hor-scroll-bar))
  (LET ((LENGTH (TRUNCATE (* width (/ (SEND self :inside-width) (SEND self :logical-width)))))
	(left (TRUNCATE (* width (/ (- x-pl-offset logical-left-edge) (SEND self :logical-width))))))
    (prepare-sheet (self)
      ;1; Erase anything there first.*
      (%draw-rectangle length scroll-bar-thickness left (- height scroll-bar-thickness)  alu-andca self)
      ;1; Now we can draw the scroll bar.*
      (%draw-rectangle length  scroll-bar-thickness left (- height scroll-bar-thickness)  alu-ior self))
    ;1; Make a record of the scroll bar so it can be erased correctly*
    (SETQ hor-scroll-bar (LIST left length scroll-bar-thickness))))

(DEFUN-METHOD 4hor-scroll-bar-erase* x-y-scroll-bars-mixin ()
  (WHEN hor-scroll-bar
    (LET ((left (FIRST hor-scroll-bar))
	  (LENGTH (SECOND hor-scroll-bar))
	  (thickness (THIRD hor-scroll-bar)))
      (prepare-sheet (self)
	(%draw-rectangle length thickness left (- height thickness) alu-andca self)
	(SEND self :refresh-margins)
	(SETQ hor-scroll-bar nil)))))

;1;--------- overview-mixin -------------------------*

(DEFFLAVOR 4overview-mixin* ((shrink-factor 1)
			   (overview-window))
	   (always-deactivate-inferiors-mixin)
  (:required-flavors essential-x-y-scrolling-mixin
		     minimum-window)
  (:gettable-instance-variables overview-window shrink-factor))

(DEFMETHOD 4(overview-mixin :after :init*) (&rest ignore)
  (SETQ overview-window (MAKE-INSTANCE 'overview-window
				       :graph-window self
				       :superior self
				       )))						    

(DEFMETHOD 4(overview-mixin :after :change-of-size-or-margins*) (&rest ignore)
  "2Set overwiew window's new edges also.*"
  (AND overview-window 
       (SEND overview-window :set-edges
	     (sheet-inside-left self)
	     (sheet-inside-top self)
	     (sheet-inside-right self)
	     (sheet-inside-bottom self))))

(DEFMETHOD 4(overview-mixin :mouse-click*) (b ignore ignore)
  (CASE b
     (#\mouse-m-2 (PROCESS-RUN-FUNCTION '(:name "3overview*" :priority 2) 
					#'(lambda (w) (SEND w :show-overview)) self))))

(DEFMETHOD 4(overview-mixin :show-overview*) ()
  (SEND overview-window :set-label "3Overview of *")
  (SEND overview-window :expose)
  (SETQ shrink-factor (MIN (/ (SEND overview-window :inside-height) (SEND self :logical-height))
			   (/ (SEND overview-window :inside-width) (SEND self :logical-width))
			   1))
  (DOLIST (item item-list)
    (SEND item :draw-self-shrunken shrink-factor overview-window))
  ;1; draw the box that represents the graph window*
  (LET* ((left (TRUNCATE (* (- x-pl-offset logical-left-edge) shrink-factor)))
	 (right (TRUNCATE (* (+ (- x-pl-offset logical-left-edge) (SEND self :inside-width)) shrink-factor)))
	 (top (TRUNCATE (* (- y-pl-offset logical-top-edge) shrink-factor)))
	 (bottom (TRUNCATE (* (+ (- y-pl-offset logical-top-edge) (SEND self :inside-height)) shrink-factor)))
	 (bwidth (- right left))
	 (bheight (- bottom top))
	 (box (SEND overview-window :box)))
    (blinker-set-cursorpos box left top)
    (blinker-set-size box bwidth bheight)
    (blinker-set-visibility box t)))

(DEFFLAVOR 4overview-window* 
	   (box ;1; is a blinker*
	    (box-dragging-on nil)
	    graph-window
	    mouse-box-x-offset
	    mouse-box-y-offset)
	   (save-superiors-bits-mixin
	    label-mixin
	    stream-mixin
	    graphics-mixin
	    minimum-window)
  (:default-init-plist
    :label nil :blinker-p nil
    :deexposed-typeout-action :permit)
  (:gettable-instance-variables box)
  (:initable-instance-variables graph-window))

(DEFFLAVOR 4hollow-rectangular-stay-inside-blinker* ()
	   (stay-inside-blinker-mixin hollow-rectangular-blinker))

(DEFMETHOD 4(overview-window :after :init*) (&rest ignore)
  (SETQ box  (make-blinker self 'hollow-rectangular-stay-inside-blinker 
				   :visibility nil)))

(DEFMETHOD 4(overview-window :who-line-documentation-string*) ()
  "2M:  Hold down to drag box;  R:  scroll to the new position.*")

(DEFMETHOD 4(overview-window :after :handle-mouse*) (&rest ignore)
  (IF (NEQ self (window-under-mouse))
      (turn-off-box-dragging)))

(DEFMETHOD 4(overview-window :mouse-click*) (button x y)
  "2If is a mouse-m then initiate the dragging*"
    (CASE button
       (#\mouse-m (turn-on-box-dragging x y) t)
       (#\mouse-r (bury-overview-n-scroll) t)))

(DEFUN-METHOD 4bury-overview-n-scroll* overview-window ()
  (LET ((logical-x (TRUNCATE (+ (/ (SEND box :x-pos) (SEND graph-window :shrink-factor))
				(SEND graph-window :logical-left-edge))))
	(logical-y (TRUNCATE (+ (/ (SEND box :y-pos) (SEND graph-window :shrink-factor))
				(SEND graph-window :logical-top-edge)))))
  ;1; We have to deactivate and scroll in another process or the mouse process sometimes*
  ;1; blocks in window lock forever.*
    (PROCESS-RUN-FUNCTION '(:name "3deactivate and scroll*" :priority 4)
			  #'(lambda (overview graph lx ly)
			      (SEND overview :deactivate)
			      (SEND graph :scroll-to lx ly))
			  self graph-window logical-x logical-y)
    ;1; Give this other process a chance.*
    (PROCESS-SLEEP 2)))

(DEFMETHOD 4(overview-window :after :mouse-moves*) (x y)
  (COND (box-dragging-on
	 (COND ((= (mouse-buttons) 2)		;1middle still down*
		(move-box-to (+ x mouse-box-x-offset) (+ y mouse-box-y-offset))
		;1; Update the offsets, they might have changed if the movement for the box was truncated.*
		(SETQ mouse-box-x-offset (- (SEND box :x-pos) (+ x (sheet-inside-left self)))
		      mouse-box-y-offset (- (SEND box :y-pos) (+ y (sheet-inside-top self))))
		)
	       (t ;1; else turn off dragging*
		(turn-off-box-dragging))))))

(DEFUN-METHOD 4move-box-to* overview-window (x y)
    (blinker-set-cursorpos box x y))
  
(DEFUN-METHOD 4turn-off-box-dragging* overview-window ()
  (SETQ box-dragging-on nil))

(DEFUN-METHOD 4turn-on-box-dragging* overview-window (x y)
  "3record the mouse-box-offsets, and set the flag*"
  ;1; (format self "~a ~a ~a ~a ~%" x y (send box :x-pos) (send box :y-pos))*
  (SETQ mouse-box-x-offset (- (SEND box :x-pos) x)
	mouse-box-y-offset (- (SEND box :y-pos) y)
	box-dragging-on t))

(DEFMETHOD 4(scrollable-line-item :draw-self-shrunken*) (shrink-factor overview-window)
  (SEND overview-window :draw-line
	(TRUNCATE (* (- from-x (SEND window :logical-left-edge)) shrink-factor))
	(TRUNCATE (* (- from-y (SEND window :logical-top-edge)) shrink-factor))
	(TRUNCATE (* (- to-x (SEND window :logical-left-edge)) shrink-factor))
	(TRUNCATE (* (- to-y (SEND window :logical-top-edge)) shrink-factor))))

(DEFMETHOD 4(scrollable-text-item :draw-self-shrunken*) (shrink-factor overview-window)
  (SEND overview-window :draw-rectangle
	(CEILING (* (- right-edge left-edge) shrink-factor))
	(CEILING (* (- bottom-edge top-edge) shrink-factor))
	(CEILING (* (- logical-x (SEND window :logical-left-edge)) shrink-factor))
	(CEILING (* (- logical-y (SEND window :logical-top-edge)) shrink-factor))))

(DEFFLAVOR 4always-deactivate-inferiors-mixin* ()
	   ())

(DEFMETHOD 4(always-deactivate-inferiors-mixin :after :deexpose*) (&rest ignore)
  (DOLIST (inferior (SEND self :inferiors))
    (SEND inferior :deactivate)))   

;1; this flavor adapted from Rich Acuff's temporary-typeout-window-mixin*

(DEFFLAVOR 4save-superiors-bits-mixin* 
	   ((bits-covered?)
	    (covered-bits))
	   ()
  (:required-flavors essential-window))


(DEFMETHOD 4(save-superiors-bits-mixin :after :deexpose*) (&rest ignore)
  "2Restore the bits of our superior.*"
  (WHEN bits-covered?
    (sheet-force-access (superior t)
      (BITBLT alu-seta width height
	      covered-bits 0 0
	      (sheet-screen-array superior) x-offset y-offset))
    (SETF bits-covered? nil)))

(DEFMETHOD 4(save-superiors-bits-mixin :before :expose*)
	   (&rest ignore)
  "2Save the bits of our superior.*"
  (IF covered-bits   ;1; Been created yet?*
      ;1; Yes, then make sure it's big enough.*
      (LET ((save-height (ARRAY-DIMENSION covered-bits 0))
	    (save-width (ARRAY-DIMENSION covered-bits 1)))
	(WHEN (OR (< save-height height)   ;1; Need to grow?*
		  (< save-width width))
	  (grow-bit-array covered-bits width height width
			  save-height save-width nil)))
     ;1; Nothing yet, make one.*
      (SETF covered-bits (make-sheet-bit-array self width height)))
  ;1; Save the old stuff if it's there.*
  (WHEN (sheet-screen-array superior)
    (prepare-sheet (superior)
      (BITBLT alu-seta width height
	    (sheet-screen-array superior) x-offset y-offset
	    covered-bits 0 0)
    ;1; Remember we've saved something.*
    (SETF bits-covered? t))))

;1; --------- mouse sensitivity -------------------------------*

(DEFFLAVOR 4mouse-sensitivity-for-instances-mixin* 
	   ((item-type-alist nil)	 ;1; Associates actions with types of items*
	    menu			 ;1; For when item clicked on with right button*
	    item-blinker 
	    (mouse-sensitive-types :all)
	    (currently-boxed-item nil))			
	   ()
  (:settable-instance-variables item-type-alist mouse-sensitive-types)
  (:initable-instance-variables item-type-alist mouse-sensitive-types)
  (:gettable-instance-variables item-blinker currently-boxed-item)
  (:required-flavors essential-x-y-scrolling-mixin))

(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :after :init*) (IGNORE)
  "2Make a pop-up menu.*"
  (setq	menu (make-window 'momentary-menu ':superior self)
	item-blinker (make-blinker self 'hollow-rectangular-blinker
				   :visibility nil)))

(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :after :handle-mouse*) (&rest ignore)
  (WHEN currently-boxed-item
    (SEND currently-boxed-item :erase-boxing)
    (SETQ currently-boxed-item nil)))

(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :current-mouse-sensitve-type-p*) (type)
  (AND (OR (EQL mouse-sensitive-types :all)
	   (AND (LISTP mouse-sensitive-types)
		(MEMBER type mouse-sensitive-types)))
       (ASSOC type item-type-alist :test #'EQ)))

(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :after :mouse-moves*) (x y)  ;1;x and y are in outside coordinates*
  "2display the boxing around the item under the mouse cursor*"
  (UNLESS (SEND self :dragging-screen-p)
    (LET ((logical-mouse-x (- (+ x x-pl-offset) (sheet-inside-left self)))
	  (logical-mouse-y (- (+ y y-pl-offset) (sheet-inside-top self))))
      (COND (;1; If scrolling is in, erase any boxing*
	     (OR hor-scrolling-in-effect ver-scrolling-in-effect)
	     (AND currently-boxed-item (SEND currently-boxed-item :erase-boxing))
	     (SETQ currently-boxed-item nil))
	    ((NOT currently-boxed-item) ;1; If nothing is boxed, look for something.*
	     (DOLIST (item item-list)
	       (IF (SEND item :draw-boxing-maybe logical-mouse-x logical-mouse-y)
		   (RETURN (SETQ currently-boxed-item item)))))
	    ((SEND currently-boxed-item :boxing-appropriate-p logical-mouse-x logical-mouse-y))	;1; then do nothing*
	    (t (SEND currently-boxed-item :erase-boxing)	;1; else we must have moved off the currently-boxed-item*
	       (SETQ currently-boxed-item nil))))))

(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :who-line-documentation-string*) ()
  (WHEN currently-boxed-item
    (LET ((DOCUMENTATION (THIRD (ASSOC (SEND currently-boxed-item :mouse-sensitive-type) item-type-alist :test #'EQ))))
      (COND ((STRINGP documentation) documentation)
	    ((CONSP   documentation)
	     (FUNCALL (CAR documentation) (SEND currently-boxed-item :item)))))))


(DEFMETHOD 4(mouse-sensitivity-for-instances-mixin :mouse-click*) (button ignore ignore)
  "2Mouse-left selects the blinking item, mouse-right pops up a menu near it.*"
  ;1; This code filched from basic-mouse-sensitive-items*
  (WHEN currently-boxed-item
    (LET ((item-type (ASSOC (SEND currently-boxed-item :mouse-sensitive-type) item-type-alist :test #'EQ)))
      (WHEN item-type
	(CASE button
	  (#\Mouse-l-1
	   ;1; Form the blip and stuff it into the keyboard buffer.*
	   (SEND self ':force-kbd-input
			 (LIST ':typeout-execute (CADR item-type) (SEND currently-boxed-item :item))))
	  (#\Mouse-r-1
	   (PROCESS-RUN-FUNCTION
	     "3Menu Choose*" #'choose-an-operation
	     menu (CDDDR item-type) self currently-boxed-item
	     ;1; Compute a label for the menu.*
	     (OR (AND (CONSP (THIRD item-type))
		      (CADR (THIRD item-type))
		      (FUNCALL (CADR (THIRD item-type))
			       (SEND currently-boxed-item :item)))))))))))



(DEFUN 4choose-an-operation* (menu alist window item menu-label)
;1; Code from typeout-menu-choose*
   "2Select a thing to do to mouse-sensitive item TYPEOUT-ITEM.
  ALIST*			2menu item-list to be displayed in MENU.
  item                    is an instance of a scrollable-item
  MENU-LABEL*		2a string to display as the menu's label, or NIL*
			2  for no label.
  The user's choice is processed by forcing input of the same sort as is
  done by clicking left on the typeout-item, ie like:*
	2(:TYPEOUT-EXECUTE operation item-information).*"
  (LET ((old-x mouse-x)	
	(old-y mouse-y))
    (FUNCALL menu ':set-label menu-label)
    (FUNCALL menu ':set-item-list alist)
    (move-window-near-rectangle menu  ;1; put the following in physcial coordinates.*
				(- (SEND item :left-edge)(SEND window :x-pl-offset))
				(- (SEND item :top-edge) (SEND window :y-pl-offset))
				(- (SEND item :right-edge) (SEND window :x-pl-offset))
				(- (SEND item :bottom-edge) (SEND window :y-pl-offset)))
    (LET ((choice-result (FUNCALL menu ':choose)))
      (AND choice-result
	   (FUNCALL window :force-kbd-input
		    (LIST :typeout-execute choice-result
			  (SEND item :item)))))
    (IGNORE old-x old-y)
    (mouse-standard-blinker window)
    (SEND window :mouse-moves mouse-x mouse-y)))

;1;------- basic-mouse-sensitive-items-compatibility-mixin ---------------*

(DEFFLAVOR 4basic-mouse-sensitive-items-compatibility-mixin* () ()
  (:required-instance-variables cursor-x cursor-y item-list))

(DEFMETHOD 4(basic-mouse-sensitive-items-compatibility-mixin  :item*) (type item &rest args)
  "2Does the same as basic-mouse-sensitive-items :item method. ie. it outputs a mouse
   sensitive item at the cursor.*"
  
  ;1; that ol' &rest bug. the lambda exp. blows up later on.*
  (SETQ args (COPY-LIST args))
  
  (COND ((NOT args)
	 (SEND self :scrollable-text-item item
	            :mouse-sensitive-type type
		    :coordinate-type :physical
		    :pre-print-item-modify-function #'(lambda (item)
							(ETYPECASE item
							  (STRING item)
							  (symbol (SYMBOL-NAME item))
							  (instance (SEND item :string-for-printing)))))
	 (PRINC item self)) ;1; Print it again just to move the cursor over.*
	(t (SEND self :scrollable-text-item item
		 :mouse-sensitive-type type
		 :coordinate-type :physical
		 :pre-print-item-modify-function
		 #'(lambda (IGNORE)
		     (APPLY #'FORMAT nil args)))
	   (APPLY #'FORMAT self args)))) ;1; Print it again just to move the cursor over.*

(DEFMETHOD 4(basic-mouse-sensitive-items-compatibility-mixin  :extended-item*) (type item &rest args)
  (APPLY self :item type item args))

(DEFMETHOD 4(basic-mouse-sensitive-items-compatibility-mixin  :after :clear-screen*) ()
  (SEND self :re-initialize))

;1;----------- ver-auto-scrolling-mixin -----------------*

(DEFFLAVOR 4ver-auto-scrolling-mixin* 
	   ((increment :half)
	    pixel-increment)
	   ()
  (:required-methods :scroll-to)
  (:required-flavors sheet)
  (:initable-instance-variables increment))

(DEFMETHOD 4(ver-auto-scrolling-mixin :after :init*) (&rest ignore)
  (SEND self :verify-n-set-pixel-increment))

(DEFMETHOD 4(ver-auto-scrolling-mixin :after :change-of-size-or-margins*) (&rest ignore)
  (SEND self :verify-n-set-pixel-increment))

(DEFMETHOD 4(ver-auto-scrolling-mixin :set-increment*) (new-increment-value)
  "2New-value can be a number in pixels, or one of the keywords:
  :whole :half :quarter. Increment is truncated to a multiple of the line height*"
  (SETQ increment new-increment-value)
  (SEND self :verify-n-set-pixel-increment))

(DEFMETHOD 4(ver-auto-scrolling-mixin :verify-n-set-pixel-increment*) ()
  (CASE increment
     (:whole (SETQ pixel-increment (SEND self :inside-height)))
     (:half (SETQ pixel-increment (TRUNCATE (SEND self :inside-height) 2)))
     (:quarter (SETQ pixel-increment (TRUNCATE (SEND self :inside-height) 4))))
  (UNLESS (< 0 pixel-increment (SEND self :inside-height))
    (CERROR "3Continue with the increment set to half the window height*"
	    "3the scrolling increment, ~a, is not between 0 and the window height, ~a.*"
	    pixel-increment (SEND self :inside-height))
    (IF (NOT (< 0 pixel-increment (SEND self :inside-height)))
	(SETQ pixel-increment (TRUNCATE (SEND self :inside-height) 2)))))

(DEFMETHOD 4(ver-auto-scrolling-mixin :end-of-page-exception*) ()
  "2Redefines sheet's :end-of-page-exception method to scroll up by increment.*"
  (COND ((NOT (ZEROP (sheet-end-page-flag)))
	 (LET ((m-vp more-vpos))
	   ;1; do the next two things instead of (sheet-home self)*
	   (SEND self :scroll-relative
		 0		              
		 pixel-increment	                      
		 nil) ;1; nil means expand the logical window*
	   (SETF (sheet-exceptions self) 0)
	   (sheet-clear-eol self)
	   ;1; Arrange for more processing next time around*
	   (COND ((NULL m-vp))			      ;1; No more processing at all*
		 ((>= m-vp 100000)		      ;1; More processing delayed?*
		  (SETQ more-vpos (- m-vp 100000)))   ;1; Cause to happen next time around.*
		 (t (SETQ more-vpos (sheet-deduce-more-vpos self))))))))

;1;-------- speed initial window creation up -------------

*(COMPILE-FLAVOR-METHODS basic-x-y-scrolling-window)